/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hive.metastore.datasource;

import java.sql.SQLException;
import java.util.Properties;
import javax.sql.DataSource;

import com.codahale.metrics.MetricRegistry;
import com.zaxxer.hikari.HikariConfig;
import com.zaxxer.hikari.HikariDataSource;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hive.common.metrics.common.Metrics;
import org.apache.hadoop.hive.common.metrics.common.MetricsFactory;
import org.apache.hadoop.hive.common.metrics.metrics2.CodahaleMetrics;
import org.apache.hadoop.hive.conf.HiveConf;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * DataSourceProvider for the HikariCP connection pool.
 */
public class HikariCPDataSourceProvider implements DataSourceProvider {

  private static final Logger LOG = LoggerFactory.getLogger(HikariCPDataSourceProvider.class);

  public static final String HIKARI = "hikaricp";
  private static final String CONNECTION_TIMEOUT_PROPERTY= HIKARI + ".connectionTimeout";

  @Override
  public DataSource create(Configuration hdpConfig) throws SQLException {

    LOG.debug("Creating Hikari connection pool for the MetaStore");

    String driverUrl = DataSourceProvider.getMetastoreJdbcDriverUrl(hdpConfig);
    String user = DataSourceProvider.getMetastoreJdbcUser(hdpConfig);
    String passwd = DataSourceProvider.getMetastoreJdbcPasswd(hdpConfig);
    int maxPoolSize = HiveConf.getIntVar(hdpConfig, HiveConf.ConfVars.METASTORE_CONNECTION_POOLING_MAX_CONNECTIONS);

    Properties properties = replacePrefix(
        DataSourceProvider.getPrefixedProperties(hdpConfig, HIKARI));
    long connectionTimeout = hdpConfig.getLong(CONNECTION_TIMEOUT_PROPERTY, 30000L);
    HikariConfig config = null;
    try {
      config = new HikariConfig(properties);
    } catch (Exception e) {
      throw new SQLException("Cannot create HikariCP configuration: ", e);
    }
    config.setMaximumPoolSize(maxPoolSize);
    config.setJdbcUrl(driverUrl);
    config.setUsername(user);
    config.setPassword(passwd);
    //https://github.com/brettwooldridge/HikariCP
    config.setConnectionTimeout(connectionTimeout);
    return new HikariDataSource(initMetrics(config));
  }

  @Override
  public boolean mayReturnClosedConnection() {
    // Only BoneCP should return true
    return false;
  }

  @Override
  public String getPoolingType() {
    return HIKARI;
  }

  private Properties replacePrefix(Properties props) {
    Properties newProps = new Properties();
    props.forEach((key,value) ->
        newProps.put(key.toString().replaceFirst(HIKARI + ".", ""), value));
    return newProps;
  }

  private static HikariConfig initMetrics(final HikariConfig config) {
    Metrics metrics = MetricsFactory.getInstance();
    if (metrics instanceof CodahaleMetrics) {
      final MetricRegistry registry = ((CodahaleMetrics) metrics).getMetricRegistry();
      if (registry != null) {
        config.setMetricRegistry(registry);
      }
    }
    return config;
  }
}
