#!/bin/bash

# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -e

usage() {
  echo "
usage: $0 <options>
  Required not-so-options:
     --distro-dir=DIR            path to distro specific files (debian/RPM)
     --build-dir=DIR             path to dist dir
     --prefix=PREFIX             path to install into

  Optional options:
     --doc-dir=DIR               path to install docs into [/usr/share/doc/solr]
     --lib-dir=DIR               path to install bits [/usr/lib/solr]
     --installed-lib-dir=DIR     path where lib-dir will end up on target system
     --bin-dir=DIR               path to install bins [/usr/bin]
     --examples-dir=DIR          path to install examples [doc-dir/examples]
     ... [ see source for more similar options ]
  "
  exit 1
}

OPTS=$(getopt \
  -n $0 \
  -o '' \
  -l 'prefix:' \
  -l 'distro-dir:' \
  -l 'doc-dir:' \
  -l 'lib-dir:' \
  -l 'installed-lib-dir:' \
  -l 'bin-dir:' \
  -l 'examples-dir:' \
  -l 'build-dir:' -- "$@")

if [ $? != 0 ] ; then
    usage
fi

eval set -- "$OPTS"
while true ; do
    case "$1" in
        --prefix)
        PREFIX=$2 ; shift 2
        ;;
        --distro-dir)
        DISTRO_DIR=$2 ; shift 2
        ;;
        --build-dir)
        BUILD_DIR=$2 ; shift 2
        ;;
        --doc-dir)
        DOC_DIR=$2 ; shift 2
        ;;
        --lib-dir)
        LIB_DIR=$2 ; shift 2
        ;;
        --installed-lib-dir)
        INSTALLED_LIB_DIR=$2 ; shift 2
        ;;
        --bin-dir)
        BIN_DIR=$2 ; shift 2
        ;;
        --examples-dir)
        EXAMPLES_DIR=$2 ; shift 2
        ;;
        --)
        shift ; break
        ;;
        *)
        echo "Unknown option: $1"
        usage
        exit 1
        ;;
    esac
done

for var in PREFIX BUILD_DIR DISTRO_DIR ; do
  if [ -z "$(eval "echo \$$var")" ]; then
    echo Missing param: $var
    usage
  fi
done

. ${DISTRO_DIR}/packaging_functions.sh

MAN_DIR=${MAN_DIR:-/usr/share/man/man1}
DOC_DIR=${DOC_DIR:-/usr/share/doc/solr}
LIB_DIR=${LIB_DIR:-/usr/lib/solr}
INSTALLED_LIB_DIR=${INSTALLED_LIB_DIR:-/usr/lib/solr}
EXAMPLES_DIR=${EXAMPLES_DIR:-$DOC_DIR/examples}
BIN_DIR=${BIN_DIR:-/usr/bin}
CONF_DIR=${CONF_DIR:-/etc/solr/conf}
DEFAULT_DIR=${ETC_DIR:-/etc/default}

VAR_DIR=$PREFIX/var

# CDH-26966 - Delete Solr data import handler jars
find ${BUILD_DIR} -name "solr-dataimporthandler*.jar" -exec rm {} \;

# CDH-68191 - Delete Solr cell jars
find ${BUILD_DIR} -name "solr-cell*.jar" -exec rm {} \;

# nocommit
install -d -m 0755 $PREFIX/$LIB_DIR
cp -ra ${BUILD_DIR}/dist/*.*ar $PREFIX/$LIB_DIR
cp -ra ${BUILD_DIR}/dist/solrj-lib $PREFIX/$LIB_DIR/lib

# Include solr upgrade tools
tar --use-compress-program pigz -xf ${BUILD_DIR}/solr-upgrade-dist.tar.gz -C $PREFIX/$LIB_DIR
chmod 755 $PREFIX/$LIB_DIR/solr-upgrade/*.sh

# copy in server folder
cp -ra ${BUILD_DIR}/server $PREFIX/$LIB_DIR

cp -ra ${BUILD_DIR}/server/solr/configsets/_default/conf $PREFIX/$LIB_DIR/coreconfig-template
cp -fa cloudera/solrconfig.xml cloudera/solrconfig.xml.secure cloudera/managed-schema $PREFIX/$LIB_DIR/coreconfig-template

#
# Normal and schemaless now share the same schema.
# The solrconfig.xml for schemaless is generated from the normal one by removing lines containing REMOVE_LINE_TO_ENABLE_SCHEMALESS
#
mkdir $PREFIX/$LIB_DIR/coreconfig-schemaless-template
cp -fa cloudera/managed-schema  $PREFIX/$LIB_DIR/coreconfig-schemaless-template/managed-schema
sed '/REMOVE_LINE_TO_ENABLE_SCHEMALESS/d' cloudera/solrconfig.xml         > $PREFIX/$LIB_DIR/coreconfig-schemaless-template/solrconfig.xml
sed '/REMOVE_LINE_TO_ENABLE_SCHEMALESS/d' cloudera/solrconfig.xml.secure  > $PREFIX/$LIB_DIR/coreconfig-schemaless-template/solrconfig.xml.secure


for DIRNAME in managedTemplate schemalessTemplate managedTemplateSecure schemalessTemplateSecure
do
  cp -ra $PREFIX/$LIB_DIR/coreconfig-template $PREFIX/$LIB_DIR/$DIRNAME
  cp -fa cloudera/configsetprops.json $PREFIX/$LIB_DIR/$DIRNAME
done

DIRNAME=managedTemplateSecure
cp -fa $PREFIX/$LIB_DIR/coreconfig-template/solrconfig.xml.secure $PREFIX/$LIB_DIR/$DIRNAME/solrconfig.xml
DIRNAME=schemalessTemplate
cp -fa $PREFIX/$LIB_DIR/coreconfig-schemaless-template/solrconfig.xml $PREFIX/$LIB_DIR/$DIRNAME/solrconfig.xml
DIRNAME=schemalessTemplateSecure
cp -fa $PREFIX/$LIB_DIR/coreconfig-schemaless-template/solrconfig.xml.secure $PREFIX/$LIB_DIR/$DIRNAME/solrconfig.xml


cp -ra cloudera/clusterconfig $PREFIX/$LIB_DIR/clusterconfig

install -d -m 0755 $PREFIX/$LIB_DIR/contrib
cp -ra ${BUILD_DIR}/contrib/velocity $PREFIX/$LIB_DIR/contrib

# Copy in the configuration files
install -d -m 0755 $PREFIX/$DEFAULT_DIR
cp $DISTRO_DIR/solr.default $PREFIX/$DEFAULT_DIR/solr

install -d -m 0755 $PREFIX/${CONF_DIR}.dist
cp -ra ${BUILD_DIR}/server/solr/* $PREFIX/${CONF_DIR}.dist
cp -fa ${BUILD_DIR}/server/scripts/cloud-scripts/log4j.properties $PREFIX/${CONF_DIR}.dist

# TODO: Solr is no longer a WAR
echo "TODO: SKIPPING install -d -m 0755 $PREFIX/$LIB_DIR/webapps/solr"
echo "TODO: SKIPPING (cd $PREFIX/$LIB_DIR/webapps/solr ; jar -xf ../../*.war)"

echo "TODO: SKIPPING cp ${BUILD_DIR}/example/lib/ext/*.jar $PREFIX/$LIB_DIR/webapps/solr/WEB-INF/lib/"

# TODO: Solr is no longer a WAR
# install -d -m 0755 $PREFIX/$LIB_DIR/webapps/ROOT
# cat > $PREFIX/$LIB_DIR/webapps/ROOT/index.html <<__EOT__
# <html><head><meta http-equiv="refresh" content="0;url=./solr"></head><body><a href="/solr">Solr Console</a></body></html>
# __EOT__
#

install -d -m 0755 $PREFIX/$LIB_DIR/server
cp -ra ${BUILD_DIR}/server/. $PREFIX/$LIB_DIR/server

cp ${BUILD_DIR}/dist/solr-depends-sentry-libs*.jar $PREFIX/$LIB_DIR/server/solr-webapp/webapp/WEB-INF/lib
cp ${BUILD_DIR}/contrib/depends-sentry-libs/lib/*.jar $PREFIX/$LIB_DIR/server/solr-webapp/webapp/WEB-INF/lib
SOLR_SENTRY_JAR=`ls $PREFIX/$LIB_DIR/server/solr-webapp/webapp/WEB-INF/lib/solr-sentry-handlers*.jar`

cp -ra ${BUILD_DIR}/dist/*.*ar $PREFIX/$LIB_DIR
mv ${BUILD_DIR}/dist/solr-sentry-audit-logging*.jar $PREFIX/$LIB_DIR/server/lib/ext
mv ${BUILD_DIR}/dist/solr-jetty-customizations*.jar $PREFIX/$LIB_DIR/server/lib/ext
cp -ra ${BUILD_DIR}/dist/solrj-lib $PREFIX/$LIB_DIR/lib

# The cloud-scripts folder contains solrctl.sh and zkcli.sh presently.
install -d -m 0755 $PREFIX/$LIB_DIR/bin

cp -ra ${BUILD_DIR}/bin/. $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/server/scripts/cloud-scripts/*.sh $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/server/scripts/cloud-scripts/log4j.properties $PREFIX/$LIB_DIR/bin
cp -a ${BUILD_DIR}/server/scripts/cloud-scripts/sentryMigrationTool $PREFIX/$LIB_DIR/bin

sed -i -e 's#/../../solr-webapp/webapp/WEB-INF/lib/#/../server/solr-webapp/webapp/WEB-INF/lib/#' $PREFIX/$LIB_DIR/bin/{zkcli.sh,sentrycli.sh,zksynctool.sh,snapshotscli.sh}
sed -i -e 's#/../../lib/ext/#/../server/lib/ext/#' $PREFIX/$LIB_DIR/bin/{zkcli.sh,sentrycli.sh,zksynctool.sh,snapshotscli.sh}

chmod 755 $PREFIX/$LIB_DIR/bin/*

mkdir "$PREFIX/$LIB_DIR/server/start.d"
chmod 777 "$PREFIX/$LIB_DIR/server/start.d"

install -d -m 0755 $PREFIX/$DOC_DIR
cp -a  ${BUILD_DIR}/*.txt $PREFIX/$DOC_DIR
cp -ra ${BUILD_DIR}/docs/* $PREFIX/$DOC_DIR
cp -ra ${BUILD_DIR}/example/ $PREFIX/$DOC_DIR/

chmod 755 $PREFIX/$LIB_DIR/bin/solrd

# Wrapper script placed under /usr/bin. Invokes $INSTALLED_LIB_DIR/bin/solrctl.sh.
install -d -m 0755 $PREFIX/$BIN_DIR
cat > $PREFIX/$BIN_DIR/solrctl <<EOF
#!/bin/bash

# Autodetect JAVA_HOME if not defined
. /usr/lib/bigtop-utils/bigtop-detect-javahome

export SOLR_HOME=${SOLR_HOME:-/usr/lib/solr/}
export SOLR_DEFAULTS=${SOLR_DEFAULTS:-/etc/default/solr}
export HADOOP_HOME=${HADOOP_HOME:-/usr/lib/hadoop/}

exec $INSTALLED_LIB_DIR/bin/solrctl.sh "\$@"

EOF
chmod 755 $PREFIX/$BIN_DIR/solrctl

# precreating /var layout
install -d -m 0755 $VAR_DIR/log/solr
install -d -m 0755 $VAR_DIR/run/solr
install -d -m 0755 $VAR_DIR/lib/solr

# Cloudera specific
install -d -m 0755 $PREFIX/$LIB_DIR/cloudera
cp cloudera/cdh_version.properties $PREFIX/$LIB_DIR/cloudera/
# tmp workaround, make logs dir
install -d -m 0755 $PREFIX/$LIB_DIR/server/logs

internal_versionless_symlinks \
    ${PREFIX}/${LIB_DIR}/*.jar \
    ${PREFIX}/${LIB_DIR}/server/solr-webapp/webapp/WEB-INF/lib/lucene*.jar \
    ${PREFIX}/${LIB_DIR}/server/solr-webapp/webapp/WEB-INF/lib/solr*.jar

external_versionless_symlinks 'search solr lucene' \
    ${PREFIX}/${LIB_DIR}/lib \
    ${PREFIX}/${LIB_DIR}/lib/solrj-lib \
    ${PREFIX}/${LIB_DIR}/server/solr-webapp/webapp/WEB-INF/lib

# Above does not handle solr-sentry-handlers, which despite its name,
# is a sentry jar
single_external_versionless_symlink $SOLR_SENTRY_JAR
