#!/bin/bash

# Licensed to the Apache Software Foundation (ASF) under one or more
# contributor license agreements.  See the NOTICE file distributed with
# this work for additional information regarding copyright ownership.
# The ASF licenses this file to You under the Apache License, Version 2.0
# (the "License"); you may not use this file except in compliance with
# the License.  You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -ex

usage() {
  echo "
usage: $0 <options>
  Required not-so-options:
     --build-dir=DIR             path to dist.dir
     --source-dir=DIR            path to package shared files dir
     --prefix=PREFIX             path to install into

  Optional options:
     --doc-dir=DIR               path to install docs into [/usr/share/doc/kafka]
     --lib-dir=DIR               path to install Kafka home [/usr/lib/kafka]
     --installed-lib-dir=DIR     path where lib-dir will end up on target system
     --examples-dir=DIR          path to install examples [doc-dir/examples]
     ... [ see source for more similar options ]
  "
  exit 1
}

OPTS=$(getopt \
  -n $0 \
  -o '' \
  -l 'prefix:' \
  -l 'doc-dir:' \
  -l 'lib-dir:' \
  -l 'installed-lib-dir:' \
  -l 'source-dir:' \
  -l 'examples-dir:' \
  -l 'build-dir:' -- "$@")

if [ $? != 0 ] ; then
    usage
fi

eval set -- "$OPTS"
while true ; do
    case "$1" in
        --prefix)
        PREFIX=$2 ; shift 2
        ;;
        --build-dir)
        BUILD_DIR=$2 ; shift 2
        ;;
        --source-dir)
        SOURCE_DIR=$2 ; shift 2
        ;;
        --doc-dir)
        DOC_DIR=$2 ; shift 2
        ;;
        --lib-dir)
        LIB_DIR=$2 ; shift 2
        ;;
        --examples-dir)
        EXAMPLES_DIR=$2 ; shift 2
        ;;
        --)
        shift ; break
        ;;
        *)
        echo "Unknown option: $1"
        usage
        exit 1
        ;;
    esac
done

for var in PREFIX BUILD_DIR SOURCE_DIR; do
  if [ -z "$(eval "echo \$$var")" ]; then
    echo Missing param: $var
    usage
  fi
done

. ${SOURCE_DIR}/packaging_functions.sh

MAN_DIR=${MAN_DIR:-/usr/share/man/man1}
DOC_DIR=${DOC_DIR:-/usr/share/doc/kafka}
LIB_DIR=${LIB_DIR:-/usr/lib/kafka}
EXAMPLES_DIR=${EXAMPLES_DIR:-$DOC_DIR/examples}
CONF_DIR=${CONF_DIR:-/etc/kafka/conf.dist}
KAFKA_HOME=${KAFKA_HOME:-/usr/lib/kafka}
ETC_KAFKA_DIR=${ETC_KAFKA_DIR:-/etc/kafka}
DIST_DIR=${BUILD_DIR}/core/build
BIN_DIR=/usr/bin
INSTALLED_KAFKA_DIR=${LIB_DIR}

install -d -m 0755 $PREFIX/$LIB_DIR
install -d -m 0755 $PREFIX/$DOC_DIR
install -d -m 0755 $PREFIX/$EXAMPLES_DIR

install -d -m 0755 $PREFIX/var/lib/kafka/
install -d -m 0755 $PREFIX/var/log/kafka/
install -d -m 0755 $PREFIX/var/run/kafka/

# Create the data directory.
install -d -m 0755 $PREFIX/var/local/kafka/data

# Copy files to the bin and other directories
# Unzip binary tarball contents to lib directory.
tar --use-compress-program pigz -xf ${BUILD_DIR}/build/kafka*.tar.gz -C ${PREFIX}/${LIB_DIR} --strip 1

#Removing:
# - javadoc scaladoc source jars from under libs.
# - bat files from under bin
rm -f ${PREFIX}/${LIB_DIR}/libs/kafka_*source.jar
rm -f ${PREFIX}/${LIB_DIR}/libs/kafka_*javadoc.jar
rm -f ${PREFIX}/${LIB_DIR}/libs/kafka_*scaladoc.jar
rm -f ${PREFIX}/${LIB_DIR}/libs/kafka_*test.jar
rm -rf ${PREFIX}/${LIB_DIR}/bin/windows

chmod 755 $PREFIX/$LIB_DIR/bin/*

# Exposing a few scripts by placing them under /usr/bin
install -d -m 0755 $PREFIX/$BIN_DIR
for file in $(ls $BUILD_DIR/bin | sed -n 's/^\(kafka-.*\)\.sh$/\1/p') zookeeper-security-migration
do
  wrapper=$PREFIX/$BIN_DIR/$file
  cat >>$wrapper <<EOF
#!/bin/bash

# Autodetect JAVA_HOME if not defined
if [ -e /usr/lib/bigtop-utils/bigtop-detect-javahome ] ; then
  . /usr/lib/bigtop-utils/bigtop-detect-javahome
fi

exec $INSTALLED_KAFKA_DIR/bin/${file}.sh "\$@"
EOF
  chmod 755 $wrapper
done

# Copy in the configuration files
install -d -m 0755 $PREFIX/$CONF_DIR
# Prune the config directory
rm -f ${PREFIX}/${LIB_DIR}/config/{consumer.properties,producer.properties,test-log4j.properties,zookeeper.properties}
cp -a ${PREFIX}/${LIB_DIR}/config/* $PREFIX/$CONF_DIR/
rm -rf ${PREFIX}/${LIB_DIR}/config
ln -s ${ETC_KAFKA_DIR}/conf $PREFIX/$LIB_DIR/config

# Copy in the defaults file
install -d -m 0755 ${PREFIX}/etc/default
cp ${SOURCE_DIR}/kafka.default ${PREFIX}/etc/default/kafka

cp ${BUILD_DIR}/{LICENSE,NOTICE} ${PREFIX}/${LIB_DIR}/

# Cloudera specific
install -d -m 0755 $PREFIX/$LIB_DIR/cloudera
cp cloudera/cdh_version.properties $PREFIX/$LIB_DIR/cloudera/
