/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.spark.sql.execution.datasources.orc

import java.io.File
import java.sql.Timestamp
import java.util.Locale

import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.fs.Path
import org.scalatest.{BeforeAndAfterAll, Ignore}

import org.apache.spark.sql.Row
import org.apache.spark.sql.internal.SQLConf
import org.apache.spark.sql.test.SharedSQLContext
import org.apache.spark.util.Utils

case class OrcData(intField: Int, stringField: String)

abstract class OrcSuite extends OrcTest with BeforeAndAfterAll {
  import testImplicits._

  var orcTableDir: File = null
  var orcTableAsDir: File = null

  protected override def beforeAll(): Unit = {
    super.beforeAll()

    orcTableAsDir = Utils.createTempDir(namePrefix = "orctests")
    orcTableDir = Utils.createTempDir(namePrefix = "orctests")

    sparkContext
      .makeRDD(1 to 10)
      .map(i => OrcData(i, s"part-$i"))
      .toDF()
      .createOrReplaceTempView("orc_temp_table")
  }

  // CDH-74083: native ORC support is disabled in CDH.
  /*
  protected def testBloomFilterCreation(bloomFilterKind: Kind) {
    val tableName = "bloomFilter"

    withTempDir { dir =>
      withTable(tableName) {
        val sqlStatement = orcImp match {
          case "native" =>
            s"""
               |CREATE TABLE $tableName (a INT, b STRING)
               |USING ORC
               |OPTIONS (
               |  path '${dir.toURI}',
               |  orc.bloom.filter.columns '*',
               |  orc.bloom.filter.fpp 0.1
               |)
            """.stripMargin
          case "hive" =>
            s"""
               |CREATE TABLE $tableName (a INT, b STRING)
               |STORED AS ORC
               |LOCATION '${dir.toURI}'
               |TBLPROPERTIES (
               |  orc.bloom.filter.columns='*',
               |  orc.bloom.filter.fpp=0.1
               |)
            """.stripMargin
          case impl =>
            throw new UnsupportedOperationException(s"Unknown ORC implementation: $impl")
        }

        sql(sqlStatement)
        sql(s"INSERT INTO $tableName VALUES (1, 'str')")

        val partFiles = dir.listFiles()
          .filter(f => f.isFile && !f.getName.startsWith(".") && !f.getName.startsWith("_"))
        assert(partFiles.length === 1)

        val orcFilePath = new Path(partFiles.head.getAbsolutePath)
        val readerOptions = OrcFile.readerOptions(new Configuration())
        val reader = OrcFile.createReader(orcFilePath, readerOptions)
        var recordReader: RecordReaderImpl = null
        try {
          recordReader = reader.rows.asInstanceOf[RecordReaderImpl]

          // BloomFilter array is created for all types; `struct`, int (`a`), string (`b`)
          val sargColumns = Array(true, true, true)
          val orcIndex = recordReader.readRowIndex(0, null, sargColumns)

          // Check the types and counts of bloom filters
          assert(orcIndex.getBloomFilterKinds.forall(_ === bloomFilterKind))
          assert(orcIndex.getBloomFilterIndex.forall(_.getBloomFilterCount > 0))
        } finally {
          if (recordReader != null) {
            recordReader.close()
          }
        }
      }
    }
  }
  */

  test("create temporary orc table") {
    checkAnswer(sql("SELECT COUNT(*) FROM normal_orc_source"), Row(10))

    checkAnswer(
      sql("SELECT * FROM normal_orc_source"),
      (1 to 10).map(i => Row(i, s"part-$i")))

    checkAnswer(
      sql("SELECT * FROM normal_orc_source where intField > 5"),
      (6 to 10).map(i => Row(i, s"part-$i")))

    checkAnswer(
      sql("SELECT COUNT(intField), stringField FROM normal_orc_source GROUP BY stringField"),
      (1 to 10).map(i => Row(1, s"part-$i")))
  }

  test("create temporary orc table as") {
    checkAnswer(sql("SELECT COUNT(*) FROM normal_orc_as_source"), Row(10))

    checkAnswer(
      sql("SELECT * FROM normal_orc_source"),
      (1 to 10).map(i => Row(i, s"part-$i")))

    checkAnswer(
      sql("SELECT * FROM normal_orc_source WHERE intField > 5"),
      (6 to 10).map(i => Row(i, s"part-$i")))

    checkAnswer(
      sql("SELECT COUNT(intField), stringField FROM normal_orc_source GROUP BY stringField"),
      (1 to 10).map(i => Row(1, s"part-$i")))
  }

  test("appending insert") {
    sql("INSERT INTO TABLE normal_orc_source SELECT * FROM orc_temp_table WHERE intField > 5")

    checkAnswer(
      sql("SELECT * FROM normal_orc_source"),
      (1 to 5).map(i => Row(i, s"part-$i")) ++ (6 to 10).flatMap { i =>
        Seq.fill(2)(Row(i, s"part-$i"))
      })
  }

  test("overwrite insert") {
    sql(
      """INSERT OVERWRITE TABLE normal_orc_as_source
        |SELECT * FROM orc_temp_table WHERE intField > 5
      """.stripMargin)

    checkAnswer(
      sql("SELECT * FROM normal_orc_as_source"),
      (6 to 10).map(i => Row(i, s"part-$i")))
  }

  test("write null values") {
    sql("DROP TABLE IF EXISTS orcNullValues")

    val df = sql(
      """
        |SELECT
        |  CAST(null as TINYINT) as c0,
        |  CAST(null as SMALLINT) as c1,
        |  CAST(null as INT) as c2,
        |  CAST(null as BIGINT) as c3,
        |  CAST(null as FLOAT) as c4,
        |  CAST(null as DOUBLE) as c5,
        |  CAST(null as DECIMAL(7,2)) as c6,
        |  CAST(null as TIMESTAMP) as c7,
        |  CAST(null as DATE) as c8,
        |  CAST(null as STRING) as c9,
        |  CAST(null as VARCHAR(10)) as c10
        |FROM orc_temp_table limit 1
      """.stripMargin)

    df.write.format("orc").saveAsTable("orcNullValues")

    checkAnswer(
      sql("SELECT * FROM orcNullValues"),
      Row.fromSeq(Seq.fill(11)(null)))

    sql("DROP TABLE IF EXISTS orcNullValues")
  }

  test("SPARK-18433: Improve DataSource option keys to be more case-insensitive") {
    val conf = spark.sessionState.conf
    val option = new OrcOptions(Map("orc.compress".toUpperCase(Locale.ROOT) -> "NONE"), conf)
    assert(option.compressionCodec == "NONE")
  }

  test("SPARK-21839: Add SQL config for ORC compression") {
    val conf = spark.sessionState.conf
    // Test if the default of spark.sql.orc.compression.codec is snappy
    assert(new OrcOptions(Map.empty[String, String], conf).compressionCodec == "SNAPPY")

    // OrcOptions's parameters have a higher priority than SQL configuration.
    // `compression` -> `orc.compression` -> `spark.sql.orc.compression.codec`
    withSQLConf(SQLConf.ORC_COMPRESSION.key -> "uncompressed") {
      assert(new OrcOptions(Map.empty[String, String], conf).compressionCodec == "NONE")
      val map1 = Map("orc.compress" -> "zlib")
      val map2 = Map("orc.compress" -> "zlib", "compression" -> "lzo")
      assert(new OrcOptions(map1, conf).compressionCodec == "ZLIB")
      assert(new OrcOptions(map2, conf).compressionCodec == "LZO")
    }

    // Test all the valid options of spark.sql.orc.compression.codec
    Seq("NONE", "UNCOMPRESSED", "SNAPPY", "ZLIB", "LZO").foreach { c =>
      withSQLConf(SQLConf.ORC_COMPRESSION.key -> c) {
        val expected = if (c == "UNCOMPRESSED") "NONE" else c
        assert(new OrcOptions(Map.empty[String, String], conf).compressionCodec == expected)
      }
    }
  }

  // CDH-74083: does not work with CDH Hive fork.
  ignore("SPARK-23340 Empty float/double array columns raise EOFException") {
    Seq(Seq(Array.empty[Float]).toDF(), Seq(Array.empty[Double]).toDF()).foreach { df =>
      withTempPath { path =>
        df.write.format("orc").save(path.getCanonicalPath)
        checkAnswer(spark.read.orc(path.getCanonicalPath), df)
      }
    }
  }

  // CDH-74083: does not work with CDH Hive fork.
  ignore("SPARK-24322 Fix incorrect workaround for bug in java.sql.Timestamp") {
    withTempPath { path =>
      val ts = Timestamp.valueOf("1900-05-05 12:34:56.000789")
      Seq(ts).toDF.write.orc(path.getCanonicalPath)
      checkAnswer(spark.read.orc(path.getCanonicalPath), Row(ts))
    }
  }
}

// CDH-74083: native ORC support is disabled in CDH.
@Ignore
class OrcSourceSuite extends OrcSuite with SharedSQLContext {

  protected override def beforeAll(): Unit = {
    super.beforeAll()

    sql(
      s"""CREATE TABLE normal_orc(
         |  intField INT,
         |  stringField STRING
         |)
         |USING ORC
         |LOCATION '${orcTableAsDir.toURI}'
       """.stripMargin)

    sql(
      s"""INSERT INTO TABLE normal_orc
         |SELECT intField, stringField FROM orc_temp_table
       """.stripMargin)

    spark.sql(
      s"""CREATE TEMPORARY VIEW normal_orc_source
         |USING ORC
         |OPTIONS (
         |  PATH '${new File(orcTableAsDir.getAbsolutePath).toURI}'
         |)
       """.stripMargin)

    spark.sql(
      s"""CREATE TEMPORARY VIEW normal_orc_as_source
         |USING ORC
         |OPTIONS (
         |  PATH '${new File(orcTableAsDir.getAbsolutePath).toURI}'
         |)
       """.stripMargin)
  }

  // CDH-74083: native ORC support is disabled in CDH.
  /*
  test("Check BloomFilter creation") {
    testBloomFilterCreation(Kind.BLOOM_FILTER_UTF8) // After ORC-101
  }
  */
}
