/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
#ifndef _DECAF_UTIL_CONCURRENT_POOLEDTHREADLISTENER_H_
#define _DECAF_UTIL_CONCURRENT_POOLEDTHREADLISTENER_H_

#include <decaf/lang/Exception.h>
#include <decaf/util/Config.h>

namespace decaf{
namespace util{
namespace concurrent{

    class PooledThread;

    /**
     * Abstract Listener Interface for users of <code>ThreadPool</code>.
     *
     * The implementor of this class receives events related to the execution and termination
     * of threads running in the ThreadPool.
     *
     * @since 1.0
     */
    class DECAF_API PooledThreadListener {
    public:

        virtual ~PooledThreadListener() {}

        /**
         * Called by a pooled thread when it is about to begin
         * executing a new task.
         * @param thread - Pointer to the Pooled Thread that is making this call
         */
        virtual void onTaskStarted( PooledThread* thread ) = 0;

        /**
         * Called by a pooled thread when it has completed a task
         * and is going back to waiting for another task to run
         * @param thread - Pointer the the Pooled Thread that is making this call.
         */
        virtual void onTaskCompleted( PooledThread* thread ) = 0;

        /**
         * Called by a pooled thread when it has encountered an exception
         * while running a user task, after receiving this notification
         * the callee should assume that the PooledThread is now no longer
         * running.
         * @param thread - Pointer to the Pooled Thread that is making this call
         * @param ex - The Exception that occurred.
         */
        virtual void onTaskException( PooledThread* thread,
                                      lang::Exception& ex) = 0;

    };

}}}

#endif /*_DECAF_UTIL_CONCURRENT_POOLEDTHREADLISTENER_H_*/
